package cz.drg.clasificator.args;

import com.google.common.collect.Lists;
import cz.drg.clasificator.CZDRGClasificator;
import cz.drg.clasificator.args.argevaluation.ArgEvaluator;
import cz.drg.clasificator.args.argevaluation.ArgEvaluatorFactory;
import cz.drg.clasificator.args.argevaluation.Evaluator;
import cz.drg.clasificator.args.argevaluation.EvaluatorFactory;
import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.readers.CsvReader;
import cz.drg.clasificator.readers.InputReader;
import cz.drg.clasificator.readers.SystemInReader;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.writers.CsvWriter;
import cz.drg.clasificator.writers.OutputWriter;
import java.io.IOException;
import java.util.ArrayList;

/**
 * Main executor of the application responsible for evaluation of program input arguments.
 * Arguments are validated based on program setting file. If no arguments for writer/reader
 * are specified program is executed with default values specified in program settings.
 * 
 * @author Pavel Jirasek
 */
public class CZDRGExecutor implements ProgramExecutor{

    private InputReader reader;
    private OutputWriter writer;
    
    private EvaluatorFactory evalFactory;
    private Evaluator currentEvaluator;
    
    @Override
    public void execute(String[] args) {
        
            checkForConfigFileChange(args);
            evalFactory = new ArgEvaluatorFactory();
            
            for (String arg : args) {
                //it is a program argument
                if(arg.startsWith("-")){

                    if(currentEvaluator != null){

                        currentEvaluator.evaluate(this);

                    }
                    currentEvaluator = evalFactory.getEvaluator(arg.substring(1));
                }
                //it is a parameter evaluator 
                else{
                    if(currentEvaluator == null){
                        stopExecution(String.format(Constants.ERR_ILLEGAL_ARG_ORDER, arg, String.join(";", args)));
                    }
                    else{
                        currentEvaluator.addParameter(arg);
                    }
                }
            }
            
            if(currentEvaluator != null){
                currentEvaluator.evaluate(this);
            }
            
            if(reader == null){setDefaultReader();}
            if(writer == null){setDefaultWriter();}
            
            reader.initialize();
            
            startEvaluation(reader, writer);
        
    }

    public void startEvaluation(InputReader reader, OutputWriter writer) {
        new CZDRGClasificator().startEvaluation(reader, writer);
    }
    
    private void setDefaultReader() {
        try {
            InputReader reader;
            
            if(System.in.available() == 0){
                reader = new CsvReader();
            }
            else{
                reader = new SystemInReader();
            }
            
            setReader(reader);
        } catch (IOException ex) {
            throw new ShutdownException(Constants.ERR_SET_DEFAULT_READER_WRITER);
        }
    }
    
    /**
     * The changed settings file needs to be processed first, because it affects
     * the way all readers/writers are initialized.
     * @param args 
     */
    private void checkForConfigFileChange(String[] args){
        ArrayList<String> argsList = Lists.newArrayList(args);
        int indexOf = argsList.indexOf("-"+ArgEvaluator.CONFIG_FILE_ARG);
        if(indexOf == -1){
            Settings.loadSettings();
            return;
        }
        
        String configFilePath = argsList.get(indexOf+1);
        Settings.loadSettings(configFilePath);
    }
    
    private void setDefaultWriter(){
        setWriter(new CsvWriter());
    }
    
    @Override
    public void setWriter(OutputWriter writer) {
        this.writer = writer;
    }

    @Override
    public void setReader(InputReader reader) {
        this.reader = reader;
    }

    @Override
    public void stopExecution(String message) {
        throw new ShutdownException(message);
    }
}
